<?php

namespace App\Modules\Blog\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Modules\Blog\Http\Resources\PostResource;
use App\Modules\Blog\Models\Category;
use App\Modules\Blog\Models\Post;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;


class PostsController extends Controller
{
    use AuthorizesRequests;
    public function createPost() : \Illuminate\Http\JsonResponse
    {
        $this->authorize('create', Post::class);

        $post = Post::create([
            'status' => 'DRAFT',
            'slug' => Carbon::now()->timestamp,
            'category_id' => NULL,
            'content_body' => [
                'en' => [
                    'preview_title' => ' ',
                    'preview_body' => ' ',
                    'preview_image' => ' ',
                    'body' => [],
                ],
                'ru' => [
                    'preview_title' => ' ',
                    'preview_body' => ' ',
                    'preview_image' => ' ',
                    'body' => [],
                ],
            ],
        ]);

        $post->slug = $post->id;
        $post->save();
        $template = ['en' => [], 'ru' => []];
        return response()->success(__('post.created'), [
            'post' => [
                'id' => $post->id,
                'status' => $post->status,
                'category' => $template,
                'tags' => $template,
                'created_at' => $post->created_at,
                'updated_at' => $post->updated_at,
                'content_body' => $post->content_body,
            ],
        ]);
    }

    public function updatePost(Request $request): \Illuminate\Http\JsonResponse
    {
        $this->authorize('update', Post::class);

        $validator = Validator::make($request->all(), [
            'id' => 'required|integer|exists:posts,id',
            'status' => 'string|in:DRAFT,PUBLISH,ARCHIVE',
            'publish_date' => 'date',
            'slug' => 'string',
            'page_header' => 'string',
            'page_description' => 'string',
            'page_og_image' => 'string',
            'author' => 'string',
            'tags' => 'array',
            'category' => 'integer|exists:categories,id',
            'content_body' => 'required|array',
        ]);

        $validator->after(function ($validator) use ($request) {
            $contentBody = $request->input('content_body');

            if (!isset($contentBody['en']) || !isset($contentBody['ru'])) {
                $validator->errors()->add('content_body', 'The content_body must contain both "en" and "ru" keys.');
                return;
            }

            $requiredFields = ['preview_title', 'preview_body', 'preview_image', 'body'];

            foreach (['en', 'ru'] as $lang) {
                foreach ($requiredFields as $field) {
                    if (!isset($contentBody[$lang][$field])) {
                        $validator->errors()->add("content_body.{$lang}.{$field}", "The {$field} field is required.");
                    }
                    if ($field === 'body') {
                        if (!is_array($contentBody[$lang][$field])) {
                            $validator->errors()->add("content_body.{$lang}.{$field}", "The {$field} must be an array.");
                        } elseif (empty($contentBody[$lang][$field])) {
                            $validator->errors()->add("content_body.{$lang}.{$field}", "The {$field} must not be empty.");
                        }
                    } else {
                        if (!is_string($contentBody[$lang][$field]) || trim($contentBody[$lang][$field]) === '') {
                            $validator->errors()->add("content_body.{$lang}.{$field}", "The {$field} must be a non-empty string.");
                        }
                    }
                }
            }
        });

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        $post = Post::find($request->id);
        if (!$post) {
            return response()->error('Post not found', [], 422);
        }

        $fieldsToUpdate = ['status', 'publish_date', 'slug', 'page_header', 'page_description', 'page_og_image', 'author'];
        foreach ($fieldsToUpdate as $field) {
            if ($request->has($field)) {
                $post->$field = $request->$field;
            }
        }

        if ($request->has('content_body')) {
            $post->content_body = json_encode($request->input('content_body'), JSON_UNESCAPED_UNICODE);
        }

        if ($request->has('tags')) {
            $post->tag_ids = json_encode($request->tags, JSON_UNESCAPED_UNICODE);
        }

        if ($request->has('category')) {
            $post->category_id = $request->category;
        }

        $post->save();
        return response()->success('Post updated successfully');
    }

    public function updatePostContent(Request $request): \Illuminate\Http\JsonResponse
    {
        $this->authorize('update', Post::class);

        $validator = Validator::make($request->all(), [
            'id' => 'required|integer|exists:posts,id',
            'content_body' => 'required|array',
        ]);

        $validator->after(function ($validator) use ($request) {
            $contentBody = $request->input('content_body');

            if (!isset($contentBody['en']) || !isset($contentBody['ru'])) {
                $validator->errors()->add('content_body', 'The content_body must contain both "en" and "ru" keys.');
            }
        });

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        $post = Post::find($request->id);
        if (!$post) {
            return response()->error('Post not found', [], 422);
        }

        $fieldsToUpdate = ['status', 'publish_date', 'slug', 'page_header', 'page_description', 'page_og_image', 'author'];
        foreach ($fieldsToUpdate as $field) {
            if ($request->has($field)) {
                $post->$field = $request->$field;
            }
        }

        if ($request->has('content_body')) {
            $post->content_body = $request->input('content_body');
        }

        if ($request->has('tags')) {
            $post->tag_ids = !empty($request->input('tags')) ? $request->tags : [];
        }

        if ($request->has('category')) {
            $post->category_id = $request->category;
        }

        $post->save();

        return response()->success('Post updated successfully');
    }

    public function deletePost($id) : \Illuminate\Http\JsonResponse
    {
        $this->authorize('delete', Post::class);

        $post = Post::find($id);

        if (!$post) {
            return response()->error('Post not found', [], 422);
        }

        $post->delete();

        return response()->success(__('post.deleted'));
    }

    public function getPosts(Request $request) : \Illuminate\Http\JsonResponse
    {
        $this->authorize('read', Post::class);

        $request->validate([
            'name' => 'string|min:1|max:255',
            'status' => ['nullable', 'string', function ($attribute, $value, $fail) {
                $allowedStatuses = ['DRAFT', 'PUBLISH', 'ARCHIVE'];
                $statuses = explode(',', $value);

                foreach ($statuses as $status) {
                    if (!in_array(trim($status), $allowedStatuses)) {
                        return $fail("Invalid status value: '{$status}'. Allowed: " . implode(', ', $allowedStatuses));
                    }
                }
            }],
            'category_id' => ['nullable', 'string', function ($attribute, $value, $fail) {
                $ids = explode(',', $value);

                if (!empty($ids) && !ctype_digit(str_replace(',', '', $value))) {
                    return $fail("The {$attribute} must contain only numbers separated by commas.");
                }

                $count = Category::whereIn('id', $ids)->count();
                if ($count !== count($ids)) {
                    return $fail("One or more category IDs are invalid.");
                }
            }],
        ]);

        $query = Post::query();

        if ($request->filled('name')) {
            $query->whereRaw(
                'LOWER(JSON_EXTRACT(content_body, "$.en.preview_title")) LIKE ?',
                ['%' . strtolower($request->input('name')) . '%']
            )->orWhereRaw(
                'LOWER(JSON_EXTRACT(content_body, "$.ru.preview_title")) LIKE ?',
                ['%' . strtolower($request->input('name')) . '%']
            );
        }

        if ($request->filled('category_id')) {
            $statuses = explode(',', $request->input('category_id'));
            $query->whereIn('category_id', $statuses);
        }

        if ($request->filled('status')) {
            $statuses = explode(',', $request->input('status'));
            $query->whereIn('status', $statuses);
        }

        if ($request->has('page')) {
            $perPage = $request->input('per_page', config('app.per_page', 2));
            $posts = $query->paginate($perPage);
        } else {
            $posts = $query->get();
        }

        $posts->transform(function ($post) {
            $post->tags = $post->tags;
            $category = Category::find($post->category_id);

            if ($category?->name) {
                $category_name = json_decode($category->name);
                $category_ru[] = ['id' => $category->id , 'name' => $category_name->ru, 'slug' => $category->slug];
                $category_en[] = ['id' => $category->id , 'name' => $category_name->en, 'slug' => $category->slug];

                $post->category = ['en' => $category_en, 'ru' => $category_ru]; // Аксессор загружает категорию
            } else {
                $post->category = ['en' => [], 'ru' => []];
            }

            return $post;
        });

        return response()->success(__('post.get_list'), PostResource::collection($posts));
    }

    public function getPostsB2c(Request $request) : \Illuminate\Http\JsonResponse
    {
        $query = Post::query();

        $query->where('status', 'PUBLISH');

        if ($request->has('page')) {
            $perPage = $request->input('per_page', config('app.per_page', 2));
            $posts = $query->paginate($perPage);
        } else {
            $posts = $query->get();
        }

        $posts->transform(function ($post) {
            $post->tags = $post->tags; // Аксессор загружает теги

            // Загрузка категории
            $category = Category::find($post->category_id);

            if ($category?->name) {
                $category_name = json_decode($category->name);
                $category_ru[] = ['id' => $category->id , 'name' => $category_name->ru, 'slug' => $category->slug];
                $category_en[] = ['id' => $category->id , 'name' => $category_name->en, 'slug' => $category->slug];

                $post->category = ['en' => $category_en, 'ru' => $category_ru]; // Аксессор загружает категорию
            } else {
                $post->category = ['en' => [], 'ru' => []];
            }

            return $post;
        });

        return response()->success('Posts list received successfully', $posts);
    }

    public function getPost($id) : \Illuminate\Http\JsonResponse
    {
        $this->authorize('read', Post::class);

        $post = Post::find($id);

        if (!$post) {
            return response()->error(__('post.not_found'), [], 422);
        }

        $post->tags = $post->tags;

        $category = Category::find($post->category_id);
        if ($category?->name) {
            $category_name = json_decode($category->name);
            $category_ru[] = ['id' => $category->id, 'name' => $category_name->ru, 'slug' => $category->slug];
            $category_en[] = ['id' => $category->id, 'name' => $category_name->en, 'slug' => $category->slug];

            $post->category = ['en' => $category_en, 'ru' => $category_ru];
        } else {
            $post->category = null;
        }

        return response()->success(__('post.get_one'), [
            'posts' => $post,
        ]);
    }

    public function getPostB2c($slug) : \Illuminate\Http\JsonResponse
    {
        $post = Post::query()->where('slug', $slug)->where('status', 'PUBLISH')->first();
        if (!$post) {
            return response()->error('Post not found', [], 422);
        }

        $post->tags = $post->tags; // Аксессор загружает теги

        $category = Category::find($post->category_id);
        if ($category?->name) {
            $category_name = json_decode($category->name);
            $category_ru[] = ['id' => $category->id, 'name' => $category_name->ru, 'slug' => $category->slug];
            $category_en[] = ['id' => $category->id, 'name' => $category_name->en, 'slug' => $category->slug];

            $post->category = ['en' => $category_en, 'ru' => $category_ru]; // Аксессор загружает категорию
        } else {
            $post->category = null;
        }

        return response()->success('Posts list received successful', [
            'posts' => $post,
        ]);
    }

    public function getPostsByCategory($category_id) : \Illuminate\Http\JsonResponse
    {
        $category = Category::find($category_id);

        if (!$category) {
            return response()->error('Category not found', [], 422);
        }

        $posts = Post::where('category_id', $category_id)->where('status', 'PUBLISH')->get();

        if ($posts->isEmpty()) {
            return response()->error('Posts not found', [], 422);
        }

        $posts->transform(function ($post) {
            $post->tags = $post->tags; // Аксессор загружает теги

            // Загрузка категории
            $category = Category::find($post->category_id);

            if ($category?->name) {
                $category_name = json_decode($category->name);
                $category_ru[] = ['id' => $category->id , 'name' => $category_name->ru, 'slug' => $category->slug];
                $category_en[] = ['id' => $category->id , 'name' => $category_name->en, 'slug' => $category->slug];

                $post->category = ['en' => $category_en, 'ru' => $category_ru]; // Аксессор загружает категорию
            } else {
                $post->category = null;
            }
            return $post;
        });

        return response()->success('Posts list received successful', [
            'posts' => $posts,
        ]);
    }

    public function getPostsByTag($tag_id) : \Illuminate\Http\JsonResponse
    {
        $posts = Post::whereJsonContains('tag_ids', (int) $tag_id)->where('status', 'PUBLISH')->get();

        if ($posts->isEmpty()) {
            return response()->error('Posts not found', [], 422);
        }

        $posts->transform(function ($post) {
            $post->tags = $post->tags; // Аксессор загружает теги

            // Загрузка категории
            $category = Category::find($post->category_id);

            if ($category?->name) {
                $category_name = json_decode($category->name);
                $category_ru[] = ['id' => $category->id , 'name' => $category_name->ru, 'slug' => $category->slug];
                $category_en[] = ['id' => $category->id , 'name' => $category_name->en, 'slug' => $category->slug];

                $post->category = ['en' => $category_en, 'ru' => $category_ru]; // Аксессор загружает категорию
            } else {
                $post->category = null;
            }
            return $post;
        });

        return response()->success('Posts list received successful', [
            'posts' => $posts,
        ]);
    }
}
